# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

Annobin is a security annotation and checking tool for binary files consisting of two main components:

1. **annocheck** - A security checker program that examines binaries and reports potential security problems
2. **Compiler plugins** - GCC, Clang, and LLVM plugins that record security information during compilation for use by annocheck

The project is hosted at https://sourceware.org/annobin/

## Build System

This project supports both **Autotools** and **Meson** build systems.

### Autotools Build

```bash
# Generate configure script (if needed)
./regenerate-configure-and-makefiles.sh

# Configure the build
./configure

# Build everything
make

# Run tests
make check

# Install
make install
```

### Meson Build

```bash
# Configure
meson setup builddir

# Build
meson compile -C builddir

# Test
meson test -C builddir

# Install
meson install -C builddir
```

### Build Options

Both build systems support the following components (all enabled by default):

- **gcc-plugin**: GCC compiler plugin for annotation
- **clang-plugin**: Clang compiler plugin for annotation
- **llvm-plugin**: LLVM compiler plugin for annotation
- **annocheck**: Binary security checker tool
- **docs**: Documentation
- **tests**: Test suite

For Autotools, disable with `--without-<component>`. For Meson, use `-D<component>=false`.

## Testing

### Running Tests

```bash
# Run all tests (Autotools)
make check

# Run all tests (Meson)
meson test -C builddir

# Run a single test (from tests/ directory)
cd tests
./compile-test
```

Tests are shell scripts in `tests/` that use `common.sh` for setup. Each test:
- Creates a temporary directory (`tmp_<test-name>`)
- Compiles test programs with the annobin plugin
- Runs annocheck to verify annotations
- Cleans up temporary files

Test environment variables (defined in `tests/common.sh`):
- `GCC`: C compiler (default: gcc)
- `PLUGIN`: Path to annobin.so plugin
- `ANNOCHECK`: Path to annocheck binary
- `READELF`, `OBJCOPY`, `STRIP`: Binary utilities

## Architecture

### Directory Structure

- `gcc-plugin/`: GCC plugin source code
  - `annobin.cc`: Main plugin implementation
  - `annobin.h`: Plugin interface and structures
  - Architecture-specific files: `x86_64.annobin.cc`, `aarch64.annobin.cc`, `arm.annobin.cc`, `powerpc.annobin.cc`, `riscv.annobin.cc`, `s390.annobin.cc`, `i686.annobin.cc`, `dummy.annobin.cc`

- `clang-plugin/`: Clang plugin source code
  - `annobin.cpp`: Clang-specific implementation

- `llvm-plugin/`: LLVM plugin source code
  - `annobin.cpp`: LLVM-specific implementation

- `annocheck/`: Security checker tool
  - `annocheck.c`: Main framework for running checkers
  - `annocheck.h`: Checker plugin interface
  - `hardened.c`: Hardened security checks (main checker, ~9000 lines)
  - `built-by.c`: Compiler detection checker
  - `notes.c`: Annotation note parsing
  - `size.c`: Size analysis checker
  - `timing.c`: Performance timing utilities
  - `libannocheck.h.in`: Library interface for embedding annocheck

- `tests/`: Test suite with shell scripts and test programs

- `doc/`: Documentation including man pages and texinfo

### Plugin Architecture

The GCC plugin (`gcc-plugin/annobin.cc`) hooks into GCC's compilation pipeline to emit security-related annotations. The plugin:

1. Registers callbacks at various compilation stages (PLUGIN_START_UNIT, PLUGIN_ALL_PASSES_START, etc.)
2. Records compiler flags, options, and code properties as notes
3. Supports two note formats: ELF notes (`.note.gnu.build.attributes`) and string sections (`.gnu.build.attributes`)
4. Uses target-specific code (e.g., `x86_64.annobin.cc`) to emit architecture-specific annotations
5. Supports section groups or link-order sections to associate notes with code

Key concepts:
- **Note format**: Controlled by `annobin_note_format` (note vs string)
- **Attach type**: How notes link to code sections (group vs link_order)
- **Function info**: `annobin_function_info` structure tracks function names, sections, symbols

### Annocheck Architecture

Annocheck (`annocheck/annocheck.c`) provides a plugin framework for binary analysis:

1. **Checker registration**: Security checkers register via `checker` structure
2. **File iteration**: Walks binaries, archives, RPMs, and directories
3. **Callbacks**: Invokes checker callbacks for files, sections, segments, symbols
4. **DWARF support**: Optional DWARF debug info parsing
5. **Multi-format support**: Handles ELF binaries, archives (.a), RPMs, and debuginfo files

The `checker` structure (defined in `annocheck.h`) provides callbacks:
- `start_file`, `end_file`: Per-file initialization/cleanup
- `check_sec`: Per-section analysis
- `check_seg`: Per-segment analysis
- `interesting_sec`: Filter sections to examine
- `process_arg`: Handle checker-specific arguments

The hardened checker (`annocheck/hardened.c`) is the primary security checker, performing comprehensive security validation including:
- Stack protection checks
- Fortification checks
- Position-independent code validation
- RELRO (relocation read-only) verification
- Control-flow integrity checks
- And many other security hardening validations

### Target-Specific Code

Each supported architecture has a dedicated source file (e.g., `x86_64.annobin.cc`) that implements:
- `annobin_save_target_specific_information()`: Capture target details during plugin init
- `annobin_get_target_pointer_size()`: Return pointer size (32 or 64 bits)
- `annobin_record_global_target_notes()`: Emit global target-specific annotations
- `annobin_target_specific_function_notes()`: Emit per-function target annotations
- `annobin_target_start_symbol_bias()`: Symbol alignment for the target

## Key Files

- `configure.ac`: Autotools configuration (version is defined here as `ANNOBIN_VERSION`)
- `meson.build`: Meson build configuration (version must match configure.ac)
- `annobin-global.h.in`: Global version and configuration header template
- `annocheck/libannocheck.h.in`: Library API header template
- `annocheck/libannocheck.pc.in`: pkg-config file template
- `tests/common.sh`: Shared test utilities and environment setup

## Version Management

The version number is defined in two places and must be kept in sync:
- `configure.ac`: `ANNOBIN_VERSION=13.02`
- `meson.build`: `version: '13.02'`

## Development Workflow

1. **Make changes** to source files
2. **Build** using either build system
3. **Run tests** with `make check` or `meson test -C builddir`
4. **Install locally** for integration testing if needed

## Debugging

- GCC plugin debugging: Use `-fplugin-arg-annobin-verbose` with GCC
- Annocheck debugging: Use `-v` (verbose) or `-vv` (very verbose) flags
- Test debugging: Tests create `tmp_<name>` directories with build artifacts

## Important Notes

- The project uses libiberty (bundled in `annocheck/libiberty/`) for utility functions
- ELF file handling uses libelf and elfutils libraries
- DWARF debugging information is optional but provides additional analysis capabilities
- The hardened checker supports colored output (can be disabled with options)
- Tests require various tools: gcc, readelf, objcopy, strip, and optionally debuginfod
